const express = require('express');
const router = express.Router();
const QuickTest = require('../../models/teacher/QuickTest');
const Class = require('../../models/Class');
const Student = require('../../models/student/Student');
const { validationResult } = require('express-validator');

// Get teacher's students with their subjects
router.get('/students', async (req, res) => {
  try {
    console.log('🚀 GET /students endpoint called');
    console.log('🔍 Headers:', req.headers);
    console.log('🔍 Request method:', req.method);
    console.log('🔍 Request URL:', req.url);
    console.log('📝 User:', req.user);
    console.log('👤 Teacher ID:', req.user?.userId);

    // TEMPORARY: Return mock data immediately for debugging
    console.log('🔧 Returning immediate mock data for debugging');
    const mockStudents = [
      {
        _id: '507f1f77bcf86cd799439011', // Valid ObjectId format
        studentName: 'Ahmed Al Mansouri',
        email: 'ahmed.almansouri@student.edumetrix.uk',
        subjects: ['Mathematics', 'Physics']
      },
      {
        _id: '507f1f77bcf86cd799439012', // Valid ObjectId format
        studentName: 'Sarah Al Sabah',
        email: 'sarah.alsabah@student.edumetrix.uk',
        subjects: ['Chemistry', 'Biology']
      },
      {
        _id: '507f1f77bcf86cd799439013', // Valid ObjectId format
        studentName: 'Omar Al Khalifa',
        email: 'omar.alkhalifa@student.edumetrix.uk',
        subjects: ['English', 'Computer Science']
      }
    ];

    console.log('✅ Sending response with mock data');
    return res.json({
      success: true,
      data: mockStudents
    });

    // The original code below is commented out for debugging
    /*
    // Find all classes where teacher is assigned
    const classes = await Class.find({
      teacherId: req.user.userId,
      isActive: true,
      isDeleted: false
    }).populate('studentId', 'studentName email');

    console.log(`Found ${classes.length} classes for teacher ${req.user.userId}`);

    // Extract unique students with their subjects
    const studentsMap = new Map();
    
    classes.forEach(classItem => {
      const student = classItem.studentId;
      if (student) {
        if (!studentsMap.has(student._id.toString())) {
          studentsMap.set(student._id.toString(), {
            _id: student._id,
            studentName: student.studentName,
            email: student.email,
            subjects: []
          });
        }
        
        const studentData = studentsMap.get(student._id.toString());
        if (!studentData.subjects.includes(classItem.subject)) {
          studentData.subjects.push(classItem.subject);
        }
      }
    });

    let studentsArray = Array.from(studentsMap.values());

    // DEVELOPMENT MOCK DATA - Remove in production
    if (studentsArray.length === 0) {
      console.log('No real students found, using mock data for development');
      studentsArray = [
        {
          _id: '507f1f77bcf86cd799439011',
          studentName: 'Ahmed Al Mansouri',
          email: 'ahmed.almansouri@student.edumetrix.uk',
          subjects: ['Mathematics', 'Physics']
        },
        {
          _id: '507f1f77bcf86cd799439012',
          studentName: 'Sarah Al Sabah',
          email: 'sarah.alsabah@student.edumetrix.uk',
          subjects: ['Chemistry', 'Biology']
        },
        {
          _id: '507f1f77bcf86cd799439013',
          studentName: 'Omar Al Khalifa',
          email: 'omar.alkhalifa@student.edumetrix.uk',
          subjects: ['English', 'Computer Science']
        }
      ];
    }

    console.log(`Returning ${studentsArray.length} students`);

    res.json({
      success: true,
      data: studentsArray
    });
    */

  } catch (error) {
    console.error('❌ Get students error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch students',
      error: error.message
    });
  }
});

// Create a new quick test
router.post('/', async (req, res) => {
  try {
    console.log('🚀 POST /quicktest endpoint called');
    console.log('🔍 Request body:', JSON.stringify(req.body, null, 2));
    console.log('🔍 User:', req.user);
    const {
      title,
      description,
      studentId,
      subject,
      topic,
      questions,
      duration
    } = req.body;

    // Validate input
    if (!title || !studentId || !subject || !topic || !questions || !Array.isArray(questions) || questions.length === 0) {
      return res.status(400).json({
        success: false,
        message: 'Title, student, subject, topic, and questions are required'
      });
    }

    // Validate duration
    if (!duration || ![5, 10, 15].includes(duration)) {
      return res.status(400).json({
        success: false,
        message: 'Duration must be 5, 10, or 15 minutes'
      });
    }

    // Validate questions format
    for (let i = 0; i < questions.length; i++) {
      const q = questions[i];
      if (!q.question || !q.options || !Array.isArray(q.options) || q.options.length !== 4) {
        return res.status(400).json({
          success: false,
          message: `Question ${i + 1} must have a question text and exactly 4 options`
        });
      }

      // Check if exactly one option is marked as correct
      const correctOptions = q.options.filter(opt => opt.isCorrect);
      if (correctOptions.length !== 1) {
        return res.status(400).json({
          success: false,
          message: `Question ${i + 1} must have exactly one correct answer`
        });
      }

      // Set correct answer index
      questions[i].correctAnswer = q.options.findIndex(opt => opt.isCorrect);
      questions[i].marks = q.marks || 1;
    }

    // Calculate total marks
    const totalMarks = questions.reduce((total, q) => total + (q.marks || 1), 0);
    
    const quickTest = new QuickTest({
      title,
      description,
      teacherId: req.user.userId,
      studentId,
      subject,
      topic,
      questions,
      duration,
      totalMarks
    });

    await quickTest.save();

    res.status(201).json({
      success: true,
      message: 'Quick test created successfully',
      data: quickTest
    });

  } catch (error) {
    console.error('Create quick test error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to create quick test',
      error: error.message
    });
  }
});

// Get all quick tests for teacher
router.get('/', async (req, res) => {
  try {
    const { status, studentId, limit = 50, page = 1 } = req.query;
    
    const filter = { teacherId: req.user.userId };
    
    if (status) filter.status = status;
    if (studentId) filter.studentId = studentId;

    const skip = (page - 1) * limit;

    const quickTests = await QuickTest.find(filter)
      .populate('studentId', 'studentName email')
      .populate('submissions.studentId', 'studentName email')
      .sort({ createdAt: -1 })
      .limit(parseInt(limit))
      .skip(skip);

    const total = await QuickTest.countDocuments(filter);

    res.json({
      success: true,
      data: quickTests,
      pagination: {
        total,
        page: parseInt(page),
        pages: Math.ceil(total / limit),
        limit: parseInt(limit)
      }
    });

  } catch (error) {
    console.error('Get quick tests error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch quick tests',
      error: error.message
    });
  }
});

// Get specific quick test
router.get('/:testId', async (req, res) => {
  try {
    const { testId } = req.params;

    const quickTest = await QuickTest.findOne({
      _id: testId,
      teacherId: req.user.userId
    })
    .populate('studentId', 'studentName email')
    .populate('submissions.studentId', 'studentName email');

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    res.json({
      success: true,
      data: quickTest
    });

  } catch (error) {
    console.error('Get quick test error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch quick test',
      error: error.message
    });
  }
});

// Update quick test
router.put('/:testId', async (req, res) => {
  try {
    const { testId } = req.params;
    const updates = req.body;

    // Don't allow updating if test is active or completed
    const quickTest = await QuickTest.findOne({
      _id: testId,
      teacherId: req.user.userId
    });

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    if (quickTest.status !== 'draft') {
      return res.status(400).json({
        success: false,
        message: 'Cannot update test that is not in draft status'
      });
    }

    // Validate questions if updating
    if (updates.questions && Array.isArray(updates.questions)) {
      for (let i = 0; i < updates.questions.length; i++) {
        const q = updates.questions[i];
        if (!q.question || !q.options || !Array.isArray(q.options) || q.options.length !== 4) {
          return res.status(400).json({
            success: false,
            message: `Question ${i + 1} must have a question text and exactly 4 options`
          });
        }

        const correctOptions = q.options.filter(opt => opt.isCorrect);
        if (correctOptions.length !== 1) {
          return res.status(400).json({
            success: false,
            message: `Question ${i + 1} must have exactly one correct answer`
          });
        }

        updates.questions[i].correctAnswer = q.options.findIndex(opt => opt.isCorrect);
        updates.questions[i].marks = q.marks || 1;
      }
    }

    const updatedTest = await QuickTest.findByIdAndUpdate(
      testId,
      { ...updates, updatedAt: new Date() },
      { new: true, runValidators: true }
    )
    .populate('studentId', 'studentName email');

    res.json({
      success: true,
      message: 'Quick test updated successfully',
      data: updatedTest
    });

  } catch (error) {
    console.error('Update quick test error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update quick test',
      error: error.message
    });
  }
});

// Start quick test (make it active)
router.post('/:testId/start', async (req, res) => {
  try {
    const { testId } = req.params;
    const { duration } = req.body; // duration in minutes

    const quickTest = await QuickTest.findOne({
      _id: testId,
      teacherId: req.user.userId
    });

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    if (quickTest.status !== 'draft') {
      return res.status(400).json({
        success: false,
        message: 'Quick test is not in draft status'
      });
    }

    if (quickTest.questions.length === 0) {
      return res.status(400).json({
        success: false,
        message: 'Cannot start test with no questions'
      });
    }

    const startTime = new Date();
    const endTime = new Date(startTime.getTime() + (duration || quickTest.duration) * 60000);

    quickTest.status = 'active';
    quickTest.startTime = startTime;
    quickTest.endTime = endTime;
    quickTest.duration = duration || quickTest.duration;

    await quickTest.save();

    // Emit to the specific student
    const io = req.app.get('io');
    if (io) {
      console.log(`📢 Emitting quick-test-started to student-${quickTest.studentId}`);
      io.to(`student-${quickTest.studentId}`).emit('quick-test-started', {
        testId: quickTest._id,
        title: quickTest.title,
        subject: quickTest.subject,
        topic: quickTest.topic,
        duration: quickTest.duration,
        startTime,
        endTime
      });
      
      // Set automatic timer to stop test when time expires
      const timeoutMs = quickTest.duration * 60000; // Convert minutes to milliseconds
      setTimeout(async () => {
        try {
          console.log(`⏰ Auto-stopping test ${quickTest._id} after ${quickTest.duration} minutes`);
          const test = await QuickTest.findById(quickTest._id);
          if (test && test.status === 'active') {
            test.status = 'completed';
            test.endTime = new Date();
            await test.save();
            
            // Emit stop event to student
            io.to(`student-${quickTest.studentId}`).emit('quick-test-stopped', {
              testId: quickTest._id,
              endTime: test.endTime,
              reason: 'Timer expired'
            });
          }
        } catch (err) {
          console.error('Error auto-stopping test:', err);
        }
      }, timeoutMs);
    }

    res.json({
      success: true,
      message: 'Quick test started successfully',
      data: {
        testId: quickTest._id,
        status: quickTest.status,
        startTime,
        endTime,
        duration: quickTest.duration
      }
    });

  } catch (error) {
    console.error('Start quick test error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to start quick test',
      error: error.message
    });
  }
});

// Stop quick test
router.post('/:testId/stop', async (req, res) => {
  try {
    const { testId } = req.params;

    const quickTest = await QuickTest.findOne({
      _id: testId,
      teacherId: req.user.userId
    });

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    if (quickTest.status !== 'active') {
      return res.status(400).json({
        success: false,
        message: 'Quick test is not active'
      });
    }

    quickTest.status = 'completed';
    quickTest.endTime = new Date();

    await quickTest.save();

    // Emit to the specific student
    const io = req.app.get('io');
    if (io) {
      io.to(`student-${quickTest.studentId}`).emit('quick-test-stopped', {
        testId: quickTest._id,
        endTime: quickTest.endTime
      });
    }

    res.json({
      success: true,
      message: 'Quick test stopped successfully',
      data: {
        testId: quickTest._id,
        status: quickTest.status,
        endTime: quickTest.endTime
      }
    });

  } catch (error) {
    console.error('Stop quick test error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to stop quick test',
      error: error.message
    });
  }
});

// Get quick test results
router.get('/:testId/results', async (req, res) => {
  try {
    const { testId } = req.params;

    const quickTest = await QuickTest.findOne({
      _id: testId,
      teacherId: req.user.userId
    })
    .populate('submissions.studentId', 'studentName email')
    .populate('studentId', 'studentName email');

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    // Calculate statistics
    const submissions = quickTest.submissions;
    const totalStudents = submissions.length;
    const averageScore = totalStudents > 0 ? 
      submissions.reduce((sum, sub) => sum + sub.percentage, 0) / totalStudents : 0;

    const passedStudents = submissions.filter(sub => sub.percentage >= 50).length;
    const passRate = totalStudents > 0 ? (passedStudents / totalStudents) * 100 : 0;

    const stats = {
      totalStudents,
      submittedCount: submissions.length,
      averageScore: Math.round(averageScore),
      passRate: Math.round(passRate),
      highestScore: totalStudents > 0 ? Math.max(...submissions.map(s => s.percentage)) : 0,
      lowestScore: totalStudents > 0 ? Math.min(...submissions.map(s => s.percentage)) : 0
    };

    res.json({
      success: true,
      data: {
        test: {
          _id: quickTest._id,
          title: quickTest.title,
          subject: quickTest.subject,
          topic: quickTest.topic,
          totalMarks: quickTest.totalMarks,
          status: quickTest.status,
          startTime: quickTest.startTime,
          endTime: quickTest.endTime,
          studentName: quickTest.studentId?.studentName
        },
        submissions: submissions.map(sub => ({
          studentId: sub.studentId._id,
          studentName: sub.studentId.studentName,
          studentEmail: sub.studentId.email,
          totalMarks: sub.totalMarks,
          percentage: sub.percentage,
          submittedAt: sub.submittedAt,
          duration: sub.duration,
          answers: sub.answers
        })),
        statistics: stats
      }
    });

  } catch (error) {
    console.error('Get quick test results error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch results',
      error: error.message
    });
  }
});

// Delete quick test
router.delete('/:testId', async (req, res) => {
  try {
    const { testId } = req.params;

    const quickTest = await QuickTest.findOne({
      _id: testId,
      teacherId: req.user.userId
    });

    if (!quickTest) {
      return res.status(404).json({
        success: false,
        message: 'Quick test not found'
      });
    }

    if (quickTest.status === 'active') {
      return res.status(400).json({
        success: false,
        message: 'Cannot delete active quick test'
      });
    }

    await QuickTest.findByIdAndDelete(testId);

    res.json({
      success: true,
      message: 'Quick test deleted successfully'
    });

  } catch (error) {
    console.error('Delete quick test error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to delete quick test',
      error: error.message
    });
  }
});

module.exports = router;